# FILE:    depth_at_slip.R
# AUTHOR:  Pete Versteegen
# DATE:    12 December 2017
# PURPOSE: Determine the water depths at each boat slip.
#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0

library(rgdal)		# Contains the read/writeOGR for reading shapefiles 
library(akima)		# For doing interpolation in a point cloud.
#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
# Exploratory Stuff
# Report on shapefile details
ogrInfo("../data/dcl_shoreline_2462", "century2462-2017")

df.slips <- read.table(file="../data/BoatSlipsSPCS.txt", header=TRUE, sep="\t")
str(df.slips)


#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
# Read the bathymetric data
df.bath <- read.table(file="../data/bathymetryV3.txt", header=TRUE, sep="")
str(df.bath)
x.bath <- df.bath$Easting
y.bath <- df.bath$Northing
z.bath <- df.bath$Depth


# df.bath20 <- read.table(file="../data/bathymetry-20.txt", header=TRUE, sep="\t")
# str(df.bath20)
# x.bath <- df.bath20$Easting
# y.bath <- df.bath20$Northing
# z.bath <- df.bath20$Depth


# df.bath100 <- read.table(file="../data/bathymetry-100.txt", header=TRUE, sep="\t")
# str(df.bath100)
# x.bath <- df.bath100$Easting
# y.bath <- df.bath100$Northing
# z.bath <- df.bath100$Depth


#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
# Read in shapefile for lake contour
shp1 <- readOGR("../data/dcl_shoreline_2462", "century2462-2017")
summary(shp1)
x.shore <- shp1$Easting
y.shore <- shp1$Northing


#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
# Read in the boat slip locations
boatslips <- read.table(file="../data/BoatSlipsSPCS.txt", header=TRUE, sep="\t")
str(boatslips)

cat("The number of boat slips = ", length(boatslips$Easting), "\n")
x.slip <- boatslips$Easting
y.slip <- boatslips$Northing


#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
# Merge the shoreline contour (depth=0) with the bathymetry points
x <- as.numeric(c(x.bath, x.shore))
y <- as.numeric(c(y.bath, y.shore))
z <- as.numeric(c(z.bath, rep(0,length(x.shore))))


#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
# Akima interpolation and evaluation at the points
z.slip <- interpp(x, y, z, x.slip, y.slip, duplicate="strip")
class(z.slip)

#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
# Find percentages of boat slips at different depths

# CEI results: The CEI bathymetry report shows that at 2458 9% of all docks are impacted; at 2457 12% are impacted, and at 2456 15% are impacted.

n.slips <- length(x.slip)

impact <- c(2458, 2457, 2456)

z.slip <- unlist(z.slip)
(z.2458 <- 100.0 * length(z.slip[z.slip < (2462-2458)]) / n.slips)
(z.2457 <- 100.0 * length(z.slip[z.slip < (2462-2457)]) / n.slips)
(z.2456 <- 100.0 * length(z.slip[z.slip < (2462-2456)]) / n.slips)

# Results for coarse bathymetry
# > (z.2458 <- 100.0 * length(z.slip[z.slip < (2462-2458)]) / n.slips)
# [1] 10.89725
# > (z.2457 <- 100.0 * length(z.slip[z.slip < (2462-2457)]) / n.slips)
# [1] 20.13622
# > (z.2456 <- 100.0 * length(z.slip[z.slip < (2462-2456)]) / n.slips)
# [1] 37.51851


# More detailed bathymetry
# > (z.2458 <- 100.0 * length(z.slip[z.slip < (2462-2458)]) / n.slips)
# [1] 2.220906
# > (z.2457 <- 100.0 * length(z.slip[z.slip < (2462-2457)]) / n.slips)
# [1] 9.831211
# > (z.2456 <- 100.0 * length(z.slip[z.slip < (2462-2456)]) / n.slips)
# [1] 25.58484


