# REF: transient 1-D conduction(2013).pdf

#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
library(VGAM)

#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
# "tran1d.m" transient,1-dimensional conduction with varying cross section area,
# nonuniform condutivity and arbitry sources. Finite volume formulation
# Originally a matlab program. (By Dr. S. Han, sep 9, 2008)
# Converted to "R" by Pete Versteegen (8/10/2017)
#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
# list of symbols:
# ac = cross-sectional area
# dt = time step
# iflag = 0 (solution converged) ,iflag=1 (solution is not converged)
# iter = iteration counter
# maxiter =  maximum iteration allowed in a time step
# n = number of control volumes
# te = temperature at new time level
# tep = projected temperature at new time level
# te0 = temperature at old time level
# tk = diffusion coefficent(thermal conductivity)
# tstop = time to stop computation
# x = independent variable (spatial coordinate), dx=delta x %funtions called in sequence:
#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0

# function grid1d.m:assigns grid system
# function inital.m:initialize the dependent variable, te and tep
# function propty.m:=evaluate thermal properties
# function boundy.m:set boundary condition for dependent variable
# function source1d.m: prescribe source terms
# function solve.m:evaluate coefficients and sources and solve
# the resulting simultaneous equations by calling function tdma
# function convcheck.m:check the convergence, te=tep?
# function figure1.m: plots line graph x vs te 

# Properties: thermal conductivity, density and specific heat
# [tk,ro,cp]=propty(n); 
propty <- function(temp){
	n <- length(temp)
	for(i in 1:n){
		tk[i] <- 401	#conductivity
		ro[i] <- 8933	#density
		cp[i] <- 383.67	#specific heat
	}
	return(data.frame(tk, ro, cp))
}

# Incorporate source terms calling source1d.
# [sp,sc]=source1d(te,n,x,dx,ac,t);
source1d <- function(te, n, x, dx, ac, t){
	m <- length(te)
	sp <- rep(0, m)
	sc <- rep(0, m)
	return(data.frame(sp, sc))
}


# Incorpolate boundary conditions calling 'boundy_modified'.
# [te,bx0,qx0c,qx0p,qx0,bx1,qx1c,qx1p,qx1]=boundy(te,tk,dx,n,t,dt);
boundy <- function(te,tk,dx,n,t,dt){
	return(data.frame(te,bx0,qx0c,qx0p,qx0,bx1,qx1c,qx1p,qx1))
}

source("../functions/credit.R")


#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0

# Specify the number of control volumes
n <- 10			# Nmber of control volumes
tl <- 1.0			# total length of medium
t.init <- 120	# Temperature set at the surface, C
t.ambient <- 20	# Initial ambient equilibrium temperature, C

# Assign time step and maximum time
dt     <- 48		# time step. set dt <- 1.0e10 for steady state (48 was default setting) Does not work for 1.0e10
tstop  <- 120	# time to stop calculation (120 was default setting)


pdf(file="../results/1_D_validation.pdf")
#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
maxiter <- 20  # maximum number of iteration in each time step. set large for steady state
np1 <- n + 1
np2 <- n + 2
np3 <- n + 3

# Weighting function f = 1 (fully-implicit)  = 1/2 (Crank-Nicolson)  = 0 (explicit)
mwrite <- 1  # if dt>tstop mwrite must be set to 0.

# Define calculation domain
delx <- tl/n  # control volume size
dx <- rep(1.0, np2) # Interval size
dx <- delx*dx # Real size of interval - vector operation

# Replace fictitious boundary volume size to small value
dx[1]   <- 1.0e-10 
dx[np2] <- 1.0e-10 

# Assign x-coordinate
print(dx)
x <- rep(0.0, np3) # Intialize with zeros
for(m in 1:np2){
	x[m + 1] <- x[m] + dx[m]	
}
print(x)

# Define cross-sectional area
ac <- rep(1.0, np3)

# Plot colors
colors <- c("green", "red", "blue")
symbol <- c(15, 17, 18)
# Time derivative test
f <- 0

# Weighting factor choice
for(j in 1:3){
	print(paste("j =", j, "  f = ", f, sep=""))
	# Prescribe intitial temperatures for all control volumes
	te0 <- vector()
	te  <- vector()
	tep <- vector()
	for(i in 1:np2){
		te0[i] <- t.ambient
		te[i]  <- te0[i]
		tep[i] <- te[i]
	}
	
	# print("te0")
	# print(te0)
	# print("te")
	# print(te)
	# print("tep")
	# print(tep)
	# ##################################################
	# Time loop begins here
	t <- 0  # starting time
	iwrite <- 1  # Print-out counter, iwrite<mwrite means skip print out
	while(t < tstop){ # calculation continues until t>tstop
		# Iteration for convergence
		iter <- 0  # set iteration counter in each time step
		iflag <- 1  # iflag <- 1 Means convergence is not reached #iteration loop for the convergence in each time step
		tk <- vector()
		ro <- vector()
		cp <- vector()
		print(paste("Time = ", t, sep=""))
		while(iflag == 1){  # "}" is at the end of program *********

			# Prescribe thermal conductivity, density and specific heat
			df.propty <- propty(te)
			tk <- df.propty$tk
			ro <- df.propty$ro
			cp <- df.propty$cp
			# print("tk")
			# print(tk)
			# print("ro")
			# print(ro)
			# print("cp")
			# print(cp)

			# Define the source values
			df.source <- source1d(te, n, x, dx, ac, t)
			sp.source <- df.source$sp
			sc.source <- df.source$sc

			# Prescribe boundary temperature
			te[1]   <- t.init	# At the left boundary given temperature
			te[np2] <- te0[np2]	# Far away boundary

			# Evaluate the diffusion conductance and source terms
			ta <- vector()
			tb <- vector()
			tc <- vector()
			td <- vector()
			for(i in 2:np1){

				# Diffusion conductance
				ke <- tk[i]*tk[i+1]*(dx[i] + dx[i+1]) / (dx[i]*tk[i+1] + dx[i+1]*tk[i])  #east interface conductivity
				de <- 2.0*ke*ac[i+1] / (dx[i] + dx[i+1])  # east side diffusion conductance #
				kw <- tk[i]*tk[i-1]*(dx[i] + dx[i-1]) / (dx[i-1]*tk[i] + dx[i]*tk[i-1])  #west interface conductivity
				dw <- 2.0*kw*ac[i] / (dx[i-1] + dx[i]) #west side diffusion conductance #
				# print(paste("ke = ", ke, "  de = ", de, "  kw = ", kw, "  dw = ", dw, sep=""))

				# Time derivative selection
				ae <- f*de
				aw <- f*dw

				# Linearlized source term evaluation
				sp  <- sp.source[i]
				sc  <- sc.source[]
				vol <- 0.5*(ac[i] + ac[i+1])*dx[i]  # mid-volume of cv
				a0  <- ro[i]*cp[i]*vol / dt  # this term is zero for steady state
				ap  <- ae + aw + a0-f*sp*vol
				b   <- sc*vol + de*(1-f)*te0[i+1] + dw*(1-f)*te0[i-1] + 
				      (a0-(1-f)*de-(1-f)*dw + sp*(1-f)*vol)*te0[i]
				# print(paste("vol =", vol, "  a0 = ", a0, "  ap = ", ap, "  b = ", b, sep=""))

				# Setting coefficients for tdma matrix
				ta[i] <- ap
				tb[i] <- ae
				tc[i] <- aw
				td[i] <- b

				# Incorporate boundary conditions
				if(i == 2){
					td[i] <- td[i] + aw*te[1]  # At x = 0
				}
				else if( i == np1){
					td[i] <- td[i] + ae*te[np2]  # At x = L
				}
			}

			#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
			# Solve the simultaneous equations by using tdma ######################
			alpa <- vector()
			beta <- vector()
			dum <- vector()
			nq <- n
			nqp1 <- nq + 1
			nqm1 <- nq - 1

			# Forward substitution
			beta[2] <- tb[2] / ta[2]
			alpa[2] <- td[2] / ta[2]
			for(i in 3:nqp1){
				beta[i] <- tb[i] / (ta[i]-tc[i]*beta[i-1])
				alpa[i] <- (td[i] + tc[i]*alpa[i-1]) / (ta[i]-tc[i]*beta[i-1])
			}
			# Backward substitution
			dum[nqp1] <- alpa[nqp1]
			for(jj in 1:nqm1){
				i <- nqp1-jj
				dum[i] <- beta[i]*dum[i+1] + alpa[i]
			}
			# End of tdma #########

			# Update the temperature
			for(i in 2:np1){
				te[i] <- dum[i]
			}

			#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
			# Check the convergence
			errote <- rep(1.0, np2)  # Initialze error
			for(i in 1:np2){
				errote[i] <- abs(te[i]-tep[i]) / te[i]
			}
			# print("te")
			# print(te)
			# print("tep")
			# print(tep)
			# print("errote")
			# print(errote)
			# print(max(errote))

			error <- 1.0e-6  # P rescribed error tolerance
			if(max(errote) > error) { # Solution not converged
				iter <- iter + 1  # Increase the iteration counter
				tep <- te  # Update the guessed value
				iflag <- 1  # Keep the flag red
			} else {
				iflag <- 0  # Solution converged, flag is green
				# tep <- te  # update the guessed value
				# iter <- 0
			}
			
		
			print(paste("Iteration # ", iter, "  maxiter = ", maxiter, sep=""))
			if(iter > maxiter){ break }  # Need to increase maxiter
			
		} # This "end" goes with the while iflag ==1 at the top******** 


		# Solution converged
		# Advance to the next time level and reinitialize the temperature
		t <- t + dt  # Increase time
		print(paste("Next time step at t = ", t, "  with time step = ", dt, sep=""))
		for(i in 1:np2){
			te0[i] <- te[i]  # Reinitialize temperature
			tep[i] <- te0[i]
		}

		# Write the results at this time?
		if(iwrite > mwrite){
			# Print the results at selected time interval
			print(paste("iwrite = ", iwrite, "   iteration number is ", iter, sep=""))
			iwrite <- 0
		}

		iwrite <- iwrite + 1
		
	} # This "end" goes with while t<tstop #################time loop

	print(paste("End of simulation with weighting factor f = ", f, sep=""))

	#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
	# Plot the result x.vs.te
	# Locate the midpoint of control volumes
	xc <- vector()
	for(i in 1:np2){
		xc[i] <- 0.5*(x[i] + x[i+1])
	}

	#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
	# Exact solution
	teexact <- vector()
	print(te[1])
	print(te[np2])
	alpha <- tk[1] / (ro[1]*cp[1])  # Thermal diffusivity
	for(i in 1:np2){
		eta <- xc[i]/sqrt(4.0*alpha*t)
		t1 <- erf(eta)
		teexact[i] <- te[1] + (te[np2]-te[1])*t1
	}

	#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
	print(paste("j = ", j, sep=""))
	if(j == 1) {
		main.text <- "Effect of Time Derivative Scheme"
		x.label <- "Distance from the wall, m"
		y.label <- "Temperature, C"
		plot(xc, teexact, type="o", main=main.text, xlab=x.label, ylab=y.label, col="black", pch=19, cex=1.25)
		grid(col="black")
	}
	points(xc, te, type="o", pch=symbol[j], col=colors[j])
	
	#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
	print("===========================================================================================")
	f <- f + 0.5

} # This goes with weighting factor f

#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
# Weighting function f = 1 (fully-implicit)  = 1/2 (Crank-Nicolson)  = 0 (explicit)
solution <- c("0 - Explicit", "0.5 - Crank-Nicolson", "1- Fully Implicit", "Analytical")
colors2 <- c(colors, "black")
symbol2 <- c(symbol, 19)
x.text <- 0.5 * tl
y.text <- 0.6 * t.init
legend("topright", pch=symbol2, col=colors2, legend=solution, title="Weighting Function, f")
text(x=x.text, y=y.text, cex=0.7, "Temperature distribution in a semi-infinite slab.\nCorresponds to the case when the temperature\n of the exposed surface of a solid medium,\nwhich is initially at temperature To,\nis suddenly raised (or lowered) to Ts at t=0\n and is maintained at that value at all times.")
credit("1-D_v2.R")

#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0

dev.off()