# REF: transient 1-D conduction(2013).pdf
# 1_D_v5.R
# Version 5 deletes the comparisons with the analytical solution,
# makes the code only fully implicit and adds the various types of boundary conditions


#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
library(VGAM)
#

#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
# "tran1d.m" transient,1-dimensional conduction with varying cross section area,
# nonuniform condutivity and arbitry sources. Finite volume formulation
# Originally a matlab program. (By Dr. S. Han, sep 9, 2008)
# Weighting function f = 1 (fully-implicit)

# This version reduced to fully implicit only!
# Converted to "R" by Pete Versteegen (8/10/2017)
#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
# List of symbols:
# ac	= cross-sectional area
# dt	= time step
# iflag	= 0 (solution converged) ,iflag=1 (solution is not converged)
# iter	= iteration counter
# maxiter	=  maximum iteration allowed in a time step
# n		= number of control volumes
# te	= temperature at new time level
# tep	= projected temperature at new time level
# te0	= temperature at old time level
# tk	= diffusion coefficent(thermal conductivity)
# tstop	= time to stop computation
# x		= independent variable (spatial coordinate), dx=delta x %funtions called in sequence:
#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
source("../functions/credit.R")


#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
# Properties: thermal conductivity, density and specific heat
propty <- function(temp, material){
	mat <- tolower(material)
	if(mat =="copper"){
		n <- length(temp)
		for(i in 1:n){
			tk[i] <- 401	#conductivity
			ro[i] <- 8933	#density
			cp[i] <- 383.67	#specific heat
		}
		return(data.frame(tk, ro, cp))
	}

	if(mat =="rock"){
		# p 6. REF: http://ceae.colorado.edu/~amadei/CVEN5768/PDF/NOTES4.pdf
		# REF: https://pubs.usgs.gov/of/1988/0441/report.pdf, p.70
		n <- length(temp)
		for(i in 1:n){
			tk[i] <- 1.0	# For most rocks, varies between 0.5 and 4.2 W/m/K; use 1.0
			ro[i] <- 2500	# varies between 1.5 and 3.3 g/cm^3  multiply by 1,000 for kg/m^3; use 2500
			cp[i] <- 750	# varies between 500 and 1000 J/kg/K; use 750
		}
		return(data.frame(tk, ro, cp))
	}

	if(mat =="granite"){
		# REF: https://pubs.usgs.gov/of/1988/0441/report.pdf, p.70
		n <- length(temp)
		for(i in 1:n){
			tk[i] <- 2.5	# W/m/K
			ro[i] <- 2670	# kg/m^3
			cp[i] <- 950	# J/kg/K
		}
		return(data.frame(tk, ro, cp))
	}
	
	tk <- 1.0
	ro <- 1.0
	cp <- 1.0
	return(data.frame(tk, ro, cp))
}

#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
# Incorporate source terms calling source1d.
source1d <- function(te, n, x, dx, ac, t){
	m <- length(te)
	sp <- rep(0, m)
	sc <- rep(0, m)
	return(data.frame(sp, sc))
}


#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
polate <- function(t, x, y) {
	# Interpolation at time "t" for boundary conditions properties
	# x, y = data with x (time) and y (property) values

	n <- length(x)
	if(t <= x[1]) { 
		yp <- y[1]
		return(yp)
	}

	if(t >= x[n]) {
		yp <- y[n]
		return(yp)
	}

	for(i in 2:n){
		if(x[i] > t ) {
			yp <- y[i-1] + (t-x[i-1]) * (y[i]-y[i-1]) / (x[i]-x[i-1])
			return(yp)
			break
		}
	}

	return(yp)
}

#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
# Boundary conditions
boundary_condition <- function(t, bc, temp.surf, hcof.surf, flux.surf, k2, dx2, temp2){
	# bc is the boundary condition type
	# bc = 1 -> temperature profile at the surface
	# bc = 2 -> convective heat transfer at the surface
	# bc = 3 -> energy flux at the surface
	# bc = 4 -> a combination of bc=2 and bc=3

	# Temperature specified at the surface
	if(bc == 1) {
		tim <- temp.surf[,1]
		tmp <- temp.surf[,2]
		
		temp.now <- polate(t, tim, tmp)
		return(temp.now)
	}

	# Heat transfer coeffience and flow temperature specified
	if(bc == 2) {
		tim <- hcof.surf[,1]
		cof <- hcof.surf[,2]
		tmp <- hcof.surf[,3]
		
		h.fluid <- polate(t, tim, cof)
		t.fluid <- polate(t, tim, tmp)
		beta <- 2.0 * k2 / (h.fluid * dx2)
		temp.now <- (t.fluid + beta * temp2) / ( 1.0 + beta)
		return(temp.now)		
	}

	# Heat flux specified at the surface
	if(bc == 3) {
		tim <- flux.surf[,1]
		flx <- flux.surf[,2]
		q.surface <- polate(t, tim, flx)
		temp.now  <- temp2 + (q.surface * dx2)/(2.0 * k2)
		return(temp.now)		
	}
	
	# Heat flux and heat transfer coefficient specified at the surface
	if(bc == 4) {
		tim <- flux.surf[,1]
		flx <- flux.surf[,2]
		q.surface <- polate(t, tim, flx)

		tim <- hcof.surf[,1]
		cof <- hcof.surf[,2]
		tmp <- hcof.surf[,3]		
		h.fluid <- polate(t, tim, cof)
		t.fluid <- polate(t, tim, tmp)
		
		beta <- 2.0 * k2 / (h.fluid * dx2)
		temp.now <- (beta * temp2 + t.fluid + q.surface / h.fluid) / (1.0 + beta)
		return(temp.now)		
	}
}


#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
# The material is copper
# # Specify the problem
# material <- "copper"
# n <- 10			# Number of control volumes
# tl <- 1.0		# Total length of medium, m
# t.init <- 120	# Temperature set at the surface, C
# t.zero <- 20	# Initial ambient equilibrium temperature, C
# # Assign time step and maximum time
# dt     <- 48		# time step. set dt <- 1.0e10 for steady state (48 was default setting) Does not work for 1.0e10
# tstop  <- 120	# time to stop calculation (120 was default setting)
# save <- TRUE
# iprint <- TRUE

#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
# Specify the problem
# The materiall is to be specified by a character string
# Current options are:
# "copper", "rock", "granite"

material <- "rock"
material <- "granite"

n <- 50			# Number of control volumes
tl <- 0.2		# Total thickness of medium, m
t.init <- 50	# Temperature set at the surface, C
t.zero  <- 15	# Initial ambient equilibrium temperature, C

# Assign time step and maximum time
# dt     <- 0.01	# time step. set dt <- 1.0e10 for steady state (48 was default setting) Does not work for 1.0e10
tstop  <- 3000.0	# time to stop calculation (120 was default setting)
n.steps = 2000
dt <- tstop/n.steps
save <- TRUE
iprint <- FALSE
iplot <- 10
# t.init <- 0.0


bc <- 1	# Specified surface temperature - specify temp.surf
bc <- 2	# Specified convective heat transfer - specify hcof.surf
# bc <- 3	# Specified heat flux
# bc <- 4	# SBoth bc=2 and bc=3


# Specify a time dependent surface temperature profile
if(bc == 1) {
	t.init  <- 50	# Temperature set at the surface, C
	t.zero  <- 15	# Initial ambient equilibrium temperature, C
	time.value <- c(0.0,  2000.0, 2001, 3000)
	temp.value <- c(50.0, 50.0,   15.0,  15.0)
	temp.surf  <- data.frame(time.value, temp.value)
}

# Specify a convective boundary condition
if(bc == 2){
	t.zero <- 50.0
	time.value <- c(0.0, tstop)
	hcof.value <- c(100.0, 100.0)
	temp.value <- c(15.0, 15.0)
	hcof.surf  <- data.frame(time.value, hcof.value, temp.value)
}

# Specify a heat flux boundary condition
if(bc == 3) {
	t.zero <- 15.0
	time.value <- c(0.0, tstop)
	flux.value <- c(800.0, 800.0)
	flux.surf  <- data.frame(time.value, flux.value)
}

# Specify a heat flux boundary condition
if(bc == 4) {
	t.zero <- 15.0
	time.value <- c(0.0, tstop)
	flux.value <- c(800.0, 800.0)
	flux.surf  <- data.frame(time.value, flux.value) 

	time.value <- c(0.0, tstop)
	hcof.value <- c(100.0, 100.0)
	temp.value <- c(15.0, 15.0)
	hcof.surf  <- data.frame(time.value, hcof.value, temp.value)
}




#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
# Define the interval at which a snapshot is to be printed
n.interval <- 10
mwrite <- tstop/n.interval  # if dt>tstop mwrite must be set to 0. Only 10 updates and plot profiles provided for.
mwrite

#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
# When satisfied with the results, do one final run and save to a pdf file
if(save) {
	file1 <- paste("../results/1_D_", material, "_temperature_", bc, ".pdf", sep="")
	pdf(file=file1)
	x.range <- c(0.0, tl)
	y.range <- c(t.zero, t.init)
	main.text <- paste("Temperature distribution in a semi-infinite slab.",
	                   "\nMaterial = ", material, ".   Boundary Condition Type = ", bc, 
	                   "\nTemperature Profiles Out To = ", tstop, " Seconds.", sep="")
	x.label <- "Distance from the wall, m"
	y.label <- "Temperature, C"
	plot(x.range, y.range, type="n", main=main.text, xlab=x.label, ylab=y.label)
	grid(col="black")
}
#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
maxiter <- 20  # maximum number of iteration in each time step. set large for steady state
np1 <- n + 1
np2 <- n + 2
np3 <- n + 3

#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
# Define calculation domain
delx <- tl / n		# control volume size
dx <- rep(delx, np2)	# Interval size
# Replace fictitious boundary volume size to small value
dx[1]   <- 1.0e-10 
dx[np2] <- 1.0e-10 

#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
# Assign x-coordinate
x <- vector()
x[1] <- 0.0
for(m in 1:np2){
	x[m + 1] <- x[m] + dx[m]	
}


# Locate the midpoint of control volumes for use in plotting x vs. te
xc <- vector()
for(i in 1:np2){
	xc[i] <- 0.5*(x[i] + x[i+1])
}

# Define cross-sectional area
ac <- rep(1.0, np3)

# Plot colors and plot symbols
library(RColorBrewer)
colors <- brewer.pal(n.interval, "Paired")
colors
colors <- rainbow(n.interval)
colors

# Vector with the times at which temperature profiles are plotted
time.plot <- vector()

# Weighting factor f = 1 (fully-implicit)
f <- 1
#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0

# Prescribe intitial temperatures for all control volumes
te0 <- vector()
te  <- vector()
tep <- vector()
for(i in 1:np2){
	te0[i] <- t.zero
	te[i]  <- te0[i]
	tep[i] <- te[i]
}

# Define thermal properties vector
tk <- vector()
ro <- vector()
cp <- vector()


#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
# Time loop begins here
t <- 0  # starting time
iwrite <- 1  # Print/Plot-out counter, 'iwrite < mwrite' means skip print out
i.plot <- 0  # Plot counter to record time and profile

while(t < tstop){ # calculation continues until 't > tstop'

	# Iteration for convergence
	iter  <- 0  # set iteration counter in each time step
	iflag <- 1  # iflag <- 1 Means convergence is not reached #iteration loop for the convergence in each time step
	if(iprint) print(paste("Time = ", t, sep=""))

	while(iflag == 1){  # "}" is at the end of program *********

		# Prescribe thermal conductivity, density and specific heat for the given material
		df.propty <- propty(te, material)
		tk <- df.propty$tk
		ro <- df.propty$ro
		cp <- df.propty$cp

		# Define the source values
		# te = current temperature values for the nodes
		# n  = number of nodes
		# x  = the distance from the surface of the nodes 
		# dx = the size of the noded
		# ac = the cross-sectional area of the noode
		# t  = the current time
		df.source <- source1d(te, n, x, dx, ac, t)
		sp.source <- df.source$sp
		sc.source <- df.source$sc

		# Prescribe boundary temperature
		# te[1]   <- temperature at the left boundary 
		te[1]   <- boundary_condition(t, bc, temp.surf, hcof.surf, flux.surf, tk[2], dx[2], te[2])	# At the left boundary given temperature
		te[np2] <- te0[np2]	# Far away boundary

		# Evaluate the diffusion conductance and source terms
		ta <- vector()
		tb <- vector()
		tc <- vector()
		td <- vector()
		for(i in 2:np1){

			# Diffusion conductance
			ke <- tk[i]*tk[i+1]*(dx[i] + dx[i+1]) / (dx[i]*tk[i+1] + dx[i+1]*tk[i])	# east interface conductivity
			de <- 2.0*ke*ac[i+1] / (dx[i] + dx[i+1])	# east side diffusion conductance #
			kw <- tk[i]*tk[i-1]*(dx[i] + dx[i-1]) / (dx[i-1]*tk[i] + dx[i]*tk[i-1])	# west interface conductivity
			dw <- 2.0*kw*ac[i] / (dx[i-1] + dx[i])	# west side diffusion conductance #

			# Time derivative selection
			ae <- f*de
			aw <- f*dw

			# Linearlized source term evaluation
			sp  <- sp.source[i]
			sc  <- sc.source[i]
			vol <- 0.5*(ac[i] + ac[i+1])*dx[i]  # mid-volume of cv
			a0  <- ro[i]*cp[i]*vol / dt  # this term is zero for steady state
			ap  <- ae + aw + a0-f*sp*vol
			b   <- sc*vol + de*(1-f)*te0[i+1] + dw*(1-f)*te0[i-1] + 
			      (a0-(1-f)*de-(1-f)*dw + sp*(1-f)*vol)*te0[i]

			# Setting coefficients for tdma matrix
			ta[i] <- ap
			tb[i] <- ae
			tc[i] <- aw
			td[i] <- b

			# Incorporate boundary conditions
			if(i == 2){
				td[i] <- td[i] + aw*te[1]	# At x = 0
			}
			else if( i == np1){
				td[i] <- td[i] + ae*te[np2]	# At x = L
			}
		}

		#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
		# Solve the simultaneous equations by using tdma ######################
		alpa <- vector()
		beta <- vector()
		dum <- vector()
		nq <- n
		nqp1 <- nq + 1
		nqm1 <- nq - 1

		# Forward substitution
		beta[2] <- tb[2] / ta[2]
		alpa[2] <- td[2] / ta[2]
		for(i in 3:nqp1){
			beta[i] <- tb[i] / (ta[i]-tc[i]*beta[i-1])
			alpa[i] <- (td[i] + tc[i]*alpa[i-1]) / (ta[i]-tc[i]*beta[i-1])
		}
		# Backward substitution
		dum[nqp1] <- alpa[nqp1]
		for(jj in 1:nqm1){
			i <- nqp1-jj
			dum[i] <- beta[i]*dum[i+1] + alpa[i]
		}

		# Update the temperature
		for(i in 2:np1){
			te[i] <- dum[i]
		}

		#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
		# Check the convergence
		errote <- rep(1.0, np2)  # Initialze error
		for(i in 1:np2){
			errote[i] <- abs(te[i]-tep[i]) / te[i]
		}

		error <- 1.0e-6  # P rescribed error tolerance
		if(max(errote) > error) { # Solution not converged
			iter <- iter + 1  # Increase the iteration counter
			tep <- te  # Update the guessed value
			iflag <- 1  # Keep the flag red
		} else {
			iflag <- 0  # Solution converged, flag is green
		}
		
	
		if(iprint) print(paste("Iteration # ", iter, "  maxiter = ", maxiter, sep=""))
		if(iter > maxiter){ break }  # Need to increase maxiter
		
	} # This "end" goes with the while iflag ==1 at the top******** 


	# Solution converged
	# Advance to the next time level and reinitialize the temperature
	t <- t + dt  # Increase time
	if(iprint) print(paste("Next time step at t = ", t, "  with time step = ", dt, sep=""))
	for(i in 1:np2){
		te0[i] <- te[i]  # Reinitialize temperature
		tep[i] <- te0[i]
	}

	# Write the results at this time?
	if(iwrite > mwrite || t-dt <= 0){
		# Print the results at selected time interval
		print(paste("iwrite = ", iwrite, "  at time = ", t, sep=""))
		# print("xc")
		# print(xc)

		i.plot <- i.plot + 1
		points(xc, te, type="o", col=colors[i.plot], pch=19, cex=0.75)
		time.plot[i.plot] <- t - dt
		print(paste("i.plot = ", i.plot, "  time.plot[i.plot] = ", time.plot[i.plot], sep=""))
		
		iwrite <- 0
	}

	iwrite <- iwrite + 1
	
} # This "end" goes with while t<tstop #################time loop

print(paste("End of simulation with weighting factor f = ", f, sep=""))

#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0

#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
i.plot <- i.plot + 1
time.plot[i.plot] <- t
points(xc, te, type="o", col=colors[i.plot], pch=19, cex=0.75)


length(time.plot)
x.text <- 0.5 * tl
y.text <- 0.6 * t.init
legend("topright", pch=19, bg="cornsilk", col=colors, legend=time.plot, title="Time Slice, sec")
credit("1-D_v2.R")

#========1=========2=========3=========4=========5=========6=========7=========8=========9=========0
warnings()
if(save) dev.off()